/**
 * \file: grl_bmpdec_png.h
 *
 * \version: $Id: grl_bmpdec_png.h,v 1.8 2010/01/14 13:43:29 tkniep Exp $ 
 *
 * This header file declares the functions that are provided by
 * the PNG Decoder Module
 *
 * \component: SVG Bitmap Decoder (SVGBMPDEC)
 *
 * \author: T. Kniep (tkniep@de.adit-jv.com)
 *
 * \copyright: (c) 2009 ADIT Corporation
 *
 ***********************************************************************/
 
#ifndef GRL_BMPDEC_PNG_H
#define GRL_BMPDEC_PNG_H

#include <adit_typedef.h>
#include <svg_bitmap_decoder.h>
#include <svg_error.h>

#ifdef __cplusplus
extern "C" {
#endif

/*******************************************************************************************
 *   Macro definitions
 *******************************************************************************************/

/** 8-bit pixel depth */
#define		GRL_BMPDEC_PNG_BIT_DEPTH8		(8)

/** 16-bit pixel depth */
#define		GRL_BMPDEC_PNG_BIT_DEPTH16		(16)

/** Standard alpha channel filler color. Use 0xff for opaque */
#define         GRL_BMPDEC_PNG_ALPHA_FILLER             (0xff)

/** Number of color channels in PLTE entries (defined by PNG spec) */
#define         GRL_BMPDEC_PNG_PLTE_CHANNELS            (3)

/** Maximum number of palette entries */
#define         GRL_BMPDEC_PNG_PLTE_MAX_ENTRIES         (256)

/** Length of PNG headers in bytes */
#define         GRL_BMPDEC_PNG_CHUNK_HDR_SIZE           (4)

/** Name of tRNS chunk */
#define         GRL_BMPDEC_PNG_TRNS_CHUNK               (void*)("tRNS")

/** Name of PLTE chunk */
#define         GRL_BMPDEC_PNG_PLTE_CHUNK               (void*)("PLTE")

/** Size of one word in PNG images */
#define         GRL_BMPDEC_PNG_WORD_SIZE                (4)


/*******************************************************************************************
 *   Type definitions
 *******************************************************************************************/


/**
 * Store palette info, used for the palette replacement feature.
 */
typedef struct {
    SVGBoolean        use_external_palette    ;         /**< Flag whether to use external palette       */
    SVGBoolean        external_PLTE_in_use    ;         /**< */
    SVGBoolean        external_TRNS_in_use    ;         /**< */
    SVGUint8          external_PLTE_length[4] ;         /**< PLTE chunk length data                     */
    SVGUint8          external_TRNS_length[4] ;         /**< TRNS chunk length data                     */
    SVGUint32         internal_PLTE_length    ;         /**< PLTE chunk length                          */
    SVGUint32         internal_TRNS_length    ;         /**< TRNS chunk length                          */
    SVGUint8          external_PLTE_CRC[4]    ;         /**< PLTE chunk CRC checksum                    */
    SVGUint8          external_TRNS_CRC[4]    ;         /**< TRNS chunk CRC checksum                    */
    SVGUint8          external_TRNS_header[4] ;         /**< TRNS chunk header data                     */
    SVGUint32         external_PLTE_entry     ;         /**< */
    SVGUint8          *external_PLTE          ;         /**< Pointer to color palette                   */
    SVGUint8          *external_TRNS          ;         /**< Pointer to alpha palette                   */
    void              *io_ptr;                          /**< */
} GRL_BMPDEC_plte_data;


/**
 * Group info needed by several functions for setting up the decoder.
 */
typedef struct _GRL_BMPDEC_png_dec_conf
{
    SVGUint8            *p_dst_mem;         /* Pointer to destination memory        */
    SVGUint32           dst_mwidth;         /* Destination memory width             */
    SVGUint32           dst_width;          /* Destination width                    */
    SVGUint32           dst_height;         /* Destination height                   */
    SVGInt32            dst_xpos;           /* Destination position x               */
    SVGInt32            dst_ypos;           /* Destination position y               */
    SVGUint8*           p_data;             /* Pointer to data                      */
    SVGBmpDecRect       src_clip;           /* Source clip rect                     */
} GRL_BMPDEC_png_dec_conf;


/*******************************************************************************************
 *   Function prototypes
 *******************************************************************************************/


/**
 * Decode PNG bitmap.
 * The PNG lib is used to decode the PNG into the specified destination.
 *
 * \param[in]   *p_decode_info          Pointer to decoding info
 *
 * \return      SVG_NO_ERROR            Decoding was successful
 * \return      SVG_OUT_OF_MEMORY       There was not enough memory for decoding
 * \return      SVG_INVALID_VALUE       Some parameter is incorrect or missing
 * \return      SVG_BMPDEC_INVALID_CLIP The clip rect is not valid
 * \return      SVG_BMPDEC_DEC_ERROR    The decoder returned an error
 *
 */
extern	SVGError GRL_BMPDEC_draw_image_PNG( GRL_BMPDEC_dec_info *p_decode_info );


/**
 * Provide information about PNG.
 * The PNG lib is used to read image information about the encoded PNG.
 *
 * \param[in]   *ctx                    Context for Bitmap Decoder
 * \param[in]   *p_image                Pointer to image definition structure.
 * \param[out]  *image_info             Pointer to image info structure containing the results
 *
 * \return      SVG_NO_ERROR            Information retrieval was successful
 * \return      SVG_INVALID_VALUE       Some parameter is incorrect or missing
 * \return      SVG_BMPDEC_DEC_ERROR    The decoder returned an error
 * 
 */
extern	SVGError GRL_BMPDEC_get_image_info_PNG( const SVGContextBmpDec  *ctx,
                                                const SVGImage          *p_image,
                                                SVGImageInfo            *image_info );
                                               

#ifdef __cplusplus
}
#endif

#endif	/* GRL_BMPDEC_PNG_H */

